%Analysis of rock surface exposure data%
%GK April 2022, based on scripts from AR%
%AM Sep 2023%

%Extracts data from bin file for the IR50, IR225 and OSL125 signals%
%Extracts depths from a txt file with the same name as the .bin file%

clear all; close all; clc; clf;

addpath('./Functions/')
addpath('./Data/')
addpath('./Bin/')

global sigmaphi0 mu sigmaphi0_225 sigmaphi0_OSL mu_OSL mu_225 ddot ddot_OSL D0 depthIR50 depthIR225 depthOSL
 
filename='R4_20112020_Miage_ME2A';
depth=textread([filename '.txt']);

%%Define sample specific model parameters
sigmaphi0=6.8*10^-9;                     %s-1
mu=3.95;                                 %mm-1 
sigmaphi0_225=6.8*10^-9;                 %s-1
mu_225=3.95;                             %mm-1
sigmaphi0_OSL=6.8*10^-9;                 %s-1
mu_OSL=3.95;                             %mm-1
ddot=4.47/(3600*24*365.25*1000);         %Gy s-1
ddot_OSL=4.47/(3600*24*365.25*1000);     %Gy s-1
D0=1066;                                 %Gy

%select paramters for data fitting
POS=1:15;                               %Position number, can be 1:15, 17:31 or 33:47
depth=depth(1:length(POS));
sigmin=6; sigmax=17;                    %Signal integration IR50 and IR225(channels)
BGmin=200; BGmax=234;                   %BG signal integration IR50 and IR225 (channels)
sigminOSL=6; sigmaxOSL=10;              %Signal integration OSL125 (channels)
BGminOSL=175; BGmaxOSL=235;             %BG signal integration OSL125 (channels)
InstErr=1.5;                            %Instrument error (%)

LxTx_IR50=zeros(length(POS),2); LxTx_IR225=LxTx_IR50; LxTx_OSL125=LxTx_IR50;

for j=1:length(POS)
    
    %read data in from bin file
    A=get_allruns([filename '.binx'],POS(j),1); %filename, position, sets
    
    %BG correct Lx and Tx signals
    for k=1:4
    Lx(j,k)=sum(A(sigmin:sigmax,k)); 
    BG(j,k)=(sum(A(BGmin:BGmax,k))/(BGmax-BGmin+1))*(sigmax-sigmin+1); 
    perrerr(j,k)=(BG(j,k)/Lx(j,k))*100;
    BGCorrLx(j,k)=Lx(j,k)-BG(j,k);
    Sigma_BG(j,k)=sqrt(BGCorrLx(j,k));
    SigPassFail(j,k)=Lx(j,k)>(3*Sigma_BG(j,k));
    
   %% Plot luminescence decay curves to control data
    figure(1) %IR50 and IR225 signals
     sgtitle(filename)
     subplot(2,2,k); axis square;
     semilogy(A(:,k)); xlim([0,250]); hold on;
   
    end
       
    print(['./Figures/' filename '_LumCurves.eps'],'-depsc');

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% 
    for k=5:6
    Lx(j,k)=sum(A(sigminOSL:sigmaxOSL,k)); 
    BG(j,k)=(sum(A(BGminOSL:BGmaxOSL,k))/(BGmaxOSL-BGminOSL+1))*(sigmaxOSL-sigminOSL+1); 
    perrerr(j,k)=(BG(j,k)/Lx(j,k))*100;
    BGCorrLx(j,k)=Lx(j,k)-BG(j,k);
    Sigma_BG(j,k)=sqrt(BGCorrLx(j,k));
    SigPassFail(j,k)=Lx(j,k)>(3*Sigma_BG(j,k));
    
     %% Plot luminescence decay curves to control data
    figure(2) %OSl125 signal
     sgtitle(filename)
     subplot(1,2,[-4+k]); axis square;
     semilogy(A(:,k)); xlim([0,250]); hold on;

   end
     

    %Calculate LxTx ratios 
    LxTx_IR50(j,1)=BGCorrLx(j,1)/BGCorrLx(j,2); 
    LxTx_IR225(j,1)=BGCorrLx(j,3)/BGCorrLx(j,4); 
    LxTx_OSL125(j,1)=BGCorrLx(j,5)/BGCorrLx(j,6);
    
    %Calculate uncertainties
    LxTx_IR50(j,2)=sqrt(((sqrt(BGCorrLx(j,1))/BGCorrLx(j,1))^2)+((sqrt(BGCorrLx(j,2))/BGCorrLx(j,2))^2)+(InstErr/100)^2);
    LxTx_IR225(j,2)=sqrt(((sqrt(BGCorrLx(j,3))/BGCorrLx(j,3))^2)+((sqrt(BGCorrLx(j,4))/BGCorrLx(j,4))^2)+(InstErr/100)^2);
    LxTx_OSL125(j,2)=sqrt(((sqrt(BGCorrLx(j,5))/BGCorrLx(j,5))^2)+((sqrt(BGCorrLx(j,6))/BGCorrLx(j,6))^2)+(InstErr/100)^2);
    
end

%Filter data for sigpassfail

SigPassFail_IR50 = SigPassFail(:,2); LxTx_IR50=LxTx_IR50(SigPassFail_IR50,:); depthIR50=depth(SigPassFail_IR50);
SigPassFail_IR225 = SigPassFail(:,4); LxTx_IR225=LxTx_IR225(SigPassFail_IR225,:); depthIR225=depth(SigPassFail_IR225);
SigPassFail_OSL125 = SigPassFail(:,6); LxTx_OSL125=LxTx_OSL125(SigPassFail_OSL125,:); depthOSL=depth(SigPassFail_OSL125);

%Normalise data relative to arbitrary threshold (record 10 onwards)
normLxTx_IR50 = LxTx_IR50./mean(LxTx_IR50(10:end,1));
normLxTx_IR225 = LxTx_IR225./mean(LxTx_IR225(10:end,1));
normLxTx_OSL125 = LxTx_OSL125./mean(LxTx_OSL125(10:end,1));

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%Stage 2 - fit with Freiseleben et al., 2015 model%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%Initial exposure and burial time guesses
t = 0.1*(3600*24*365.25*1000);                          %time, yr
tb = 0.1*(3600*24*365.25*1000);                         %time, yr
wIR50 = ones(length(depthIR50),1); wIR50(1:2)=20; wIR50(3:4)=10;        %weight first two data points + and the third one
wIR225 = ones(length(depthIR225),1); wIR225(1:2)=100; wIR225(3)=10;     %weight first two data points + and the third one
wOSL = ones(length(depthOSL),1); wOSL(1:2)=100; wOSL(3)=10;             %weight first two data points + and the third one
%beta0=[t tb sigmaphi0 mu];                                       
beta0=[t tb];                                       

%fit data
[beta1,R,J,Cov,MSE]=nlinfit(depthIR50,normLxTx_IR50(:,1),@Freiesleben,beta0,'Weights',wIR50);      %does fit
[Ypred,delta]=nlpredci(@Freiesleben,depthIR50,beta1,R,'covar',Cov,'alpha',.05,'predopt','curve');  %predicts fit
sbeta1 = nlparci(beta1,R,'covar',Cov,'alpha',1-.68);

[beta1_IR225,R_IR225,J_IR225,Cov_IR225,MSE_IR225]=nlinfit(depthIR225,normLxTx_IR225(:,1),@Freiesleben_225,beta0,'Weights',wIR225);     %does fit
[Ypred_IR225,delta_IR225]=nlpredci(@Freiesleben_225,depthIR225,beta1_IR225,R,'covar',Cov,'alpha',.05,'predopt','curve');               %predicts fit
sbeta1_IR225 = nlparci(beta1_IR225,R,'covar',Cov,'alpha',1-.68);

[beta1_OSL,R_OSL,J_IROSL,Cov_IROSL,MSE_IROSL]=nlinfit(depthOSL,normLxTx_OSL125(:,1),@Freiesleben_OSL,beta0,'Weights',wOSL);            %does fit
[Ypred_OSL,delta_OSL]=nlpredci(@Freiesleben_OSL,depthOSL,beta1_OSL,R,'covar',Cov,'alpha',.05,'predopt','curve');                       %predicts fit
sbeta1_OSL = nlparci(beta1_OSL,R,'covar',Cov,'alpha',1-.68);

%predict exposed curve
L_exp_IR50 = 1*exp(-sigmaphi0*beta1(1)*exp(-mu*depthIR50));
L_exp_IR225 = 1*exp(-sigmaphi0_225*beta1_IR225(1)*exp(-mu_225*depthIR225));
L_exp_OSL = 1*exp(-sigmaphi0_OSL*beta1_OSL(1)*exp(-mu_OSL*depthOSL));

%calculate bleaching ratio to evaluate if data are usable cf. Rades et al.
%(2018)
BR_IR50 = round(L_exp_IR50,2)./round(Ypred,2);
BR_IR225 = round(L_exp_IR225,2)./round(Ypred_IR225,2);
BR_OSL = round(L_exp_OSL,2)./round(Ypred_OSL,2);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
figure('Position',[10 10 900 300]) 
% Plot all 3 signals together for each sample and the bleaching plateau test
% sgtitle([filename])

%IR50
subplot(3,3,[1 4])
plot(depthIR50,Ypred,'k'); hold on; 
errorbar(depthIR50,normLxTx_IR50(:,1),normLxTx_IR50(:,2),'LineStyle','none','Color','k')
plot(depthIR50,normLxTx_IR50(:,1),'o','MarkerEdgeColor','k','MarkerFaceColor','k');
if beta1~=beta0 
    if beta1(2)>=0 
        plot(depthIR50,L_exp_IR50,'-.','Color','k');
    end
end

ylabel('Luminescence Signal [a.u.]');
% subtitle('IR50')

subplot(3,3,7)
if beta1~=beta0 
    if beta1(2)>=0 
        plot(depthIR50,BR_IR50,'o','MarkerEdgeColor','k','MarkerFaceColor','w'); hold on;
    end
end
plot(depthIR50,repmat(0.05,length(depthIR50)),'--','Color','k')
ylim([0 1])
xlabel('Depth [mm]'); 
ylabel('L_{exp}/L_{bur}');


%IR225
subplot(3,3,[2 5])
plot(depthIR225,Ypred_IR225,'k'); hold on; 
errorbar(depthIR225,normLxTx_IR225(:,1),normLxTx_IR225(:,2),'LineStyle','none','Color','k')
plot(depthIR225,normLxTx_IR225(:,1),'o','MarkerEdgeColor','k','MarkerFaceColor','k');

if beta1_IR225~=beta0 
    if beta1_IR225(2)>=0 
        plot(depthIR225,L_exp_IR225,'-.','Color','k');
    end
end
% subtitle('IR225')

subplot(3,3,8)
if beta1_IR225~=beta0 
    if beta1_IR225(2)>=0
plot(depthIR225,BR_IR225,'o','MarkerEdgeColor','k','MarkerFaceColor','w'); hold on;
    end
end
plot(depthIR225,repmat(0.05,length(depthIR225)),'--','Color','k')
ylim([0 1])
xlabel('Depth [mm]'); 

%OSL125
subplot(3,3,[3 6])
plot(depthOSL,Ypred_OSL,'k'); hold on; 
errorbar(depthOSL,normLxTx_OSL125(:,1),normLxTx_OSL125(:,2),'LineStyle','none','Color','k')
plot(depthOSL,normLxTx_OSL125(:,1),'o','MarkerEdgeColor','k','MarkerFaceColor','k');
if beta1_OSL~=beta0 
    if beta1_OSL(2)>=0 
        plot(depthOSL,L_exp_OSL,'-.','Color','k');
    end
end
% subtitle('OSL125')

subplot(3,3,9)
if beta1_OSL~=beta0 
    if beta1_OSL(2)>=0 
        plot(depthOSL,BR_OSL,'o','MarkerEdgeColor','k','MarkerFaceColor','w'); hold on;
    end
end
plot(depthOSL,repmat(0.05,length(depthOSL)),'--','Color','k')
ylim([0 1])
xlabel('Depth [mm]'); 
%ylabel('L_exp/L_bur');

print(['./Figures/' filename '_IR50_IR225_OSL125.eps'],'-depsc');

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%output results
Results=[beta1; beta1_IR225; beta1_OSL]

records.teIR50 = [beta1(1) sbeta1(1,2)-beta1(1)];
records.tbIR50 = [beta1(2) sbeta1(2,2)-beta1(2)];
records.teIR225 = [beta1_IR225(1) sbeta1_IR225(1,2)-beta1_IR225(1)];
records.tbIR225 = [beta1_IR225(2) sbeta1_IR225(2,2)-beta1_IR225(2)];
records.teOSL = [beta1_OSL(1) sbeta1_OSL(1,2)-beta1_OSL(1)];
records.tbOSL = [beta1_OSL(2) sbeta1_OSL(2,2)-beta1_OSL(2)];
save(['./ComputeData/' filename '_fitpar.mat'],'records')